/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Workfile: ToolkitSample.c $
   Last Modification:
    $Author: stephans $
    $Modtime: $
    $Revision: 6511 $
   
   Targets:
     MQX   : yes
 
   Description:
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     1         11.10.2011  SS       Initial version derived from windows example
     
**************************************************************************************/
#include <string.h>
#include <mqx.h>
#include <fio.h>
#include "cifXToolkit.h"
#include "CIFXErrors.h"
#include "SerialDPMInterface.h"
#include "MQXParDPMInterface.h"
#include "ToolkitSample.h"

extern uint32_t g_ulTraceLevel;

/*****************************************************************************/
/*! Thread for cyclic Toolkit timer, which handles polling of COS bits on
 *   non-irq cards
 *     \param arg      Pollinterval in ms
 *     \return NULL on termination                                            */
/*****************************************************************************/
void cifXPollingThread(uint_32 ulParam)
{
  uint32_t ulPollInterval = ulParam;
  
  while (1)
  {
    cifXTKitCyclicTimer();
    OS_Sleep(ulPollInterval);
  }
}

/*****************************************************************************/
/*! Displays a hex dump on the debug console (16 bytes per line)
*   \param pbData     Pointer to dump data
*   \param ulDataLen  Length of data dump                                    */
/*****************************************************************************/
static void DumpData(uint8_t* pbData, uint32_t ulDataLen)
{
  uint32_t ulIdx = 0;
  for(ulIdx = 0; ulIdx < ulDataLen; ++ulIdx)
  {
    if(0 == (ulIdx % 16))
      printf("\r\n");

    printf("%02X ", pbData[ulIdx]);
  }
  printf("\r\n");
}

/*****************************************************************************/
/*! Dumps a rcX packet to debug console
*   \param ptPacket Pointer to packed being dumped                           */
/*****************************************************************************/
static void DumpPacket(CIFX_PACKET* ptPacket)
{
  printf("Dest   : 0x%08X      ID   : 0x%08X\r\n", (int)ptPacket->tHeader.ulDest,   (int)ptPacket->tHeader.ulId);
  printf("Src    : 0x%08X      Sta  : 0x%08X\r\n", (int)ptPacket->tHeader.ulSrc,    (int)ptPacket->tHeader.ulState);
  printf("DestID : 0x%08X      Cmd  : 0x%08X\r\n", (int)ptPacket->tHeader.ulDestId, (int)ptPacket->tHeader.ulCmd);
  printf("SrcID  : 0x%08X      Ext  : 0x%08X\r\n", (int)ptPacket->tHeader.ulSrcId,  (int)ptPacket->tHeader.ulExt);
  printf("Len    : 0x%08X      Rout : 0x%08X\r\n", (int)ptPacket->tHeader.ulLen,    (int)ptPacket->tHeader.ulRout);

  printf("Data:");
  DumpData(ptPacket->abData, ptPacket->tHeader.ulLen);
}

/*****************************************************************************/
/*! Function to demonstrate the board/channel enumeration
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t EnumBoardDemo(void)
{
  CIFXHANDLE hDriver = NULL;
  int32_t    lRet    = xDriverOpen(&hDriver);

  printf("\r\n---------- Board/Channel enumeration demo ----------\r\n");

  if(CIFX_NO_ERROR == lRet)
  {
    /* Driver/Toolkit successfully opened */
    uint32_t          ulBoard    = 0;
    BOARD_INFORMATION tBoardInfo = {0};

    /* Iterate over all boards */
    while(CIFX_NO_ERROR == xDriverEnumBoards(hDriver, ulBoard, sizeof(tBoardInfo), &tBoardInfo))
    {
      uint32_t            ulChannel    = 0;
      CHANNEL_INFORMATION tChannelInfo = {{0}};
   	  
      printf("Found Board %s\r\n", tBoardInfo.abBoardName);
      if(strlen( (char*)tBoardInfo.abBoardAlias) != 0)
        printf(" Alias        : %s\r\n", tBoardInfo.abBoardAlias);

      printf(" DeviceNumber : %u\r\n", (int)tBoardInfo.tSystemInfo.ulDeviceNumber);
      printf(" SerialNumber : %u\r\n", (int)tBoardInfo.tSystemInfo.ulSerialNumber);
      printf(" Board ID     : %u\r\n", (int)tBoardInfo.ulBoardID);
      printf(" System Error : 0x%08X\r\n", (int)tBoardInfo.ulSystemError);
      printf(" Channels     : %u\r\n", (int)tBoardInfo.ulChannelCnt);
      printf(" DPM Size     : %u\r\n", (int)tBoardInfo.ulDpmTotalSize);

      /* iterate over all channels on the current board */
      while(CIFX_NO_ERROR == xDriverEnumChannels(hDriver, ulBoard, ulChannel, sizeof(tChannelInfo), &tChannelInfo))
      {
        printf(" - Channel %u:\r\n", (int)ulChannel);
        printf("    Firmware : %s\r\n", tChannelInfo.abFWName);
        printf("    Version  : %u.%u.%u build %u\r\n", 
               tChannelInfo.usFWMajor,
               tChannelInfo.usFWMinor,
               tChannelInfo.usFWRevision,
               tChannelInfo.usFWBuild);
        printf("    Date     : %02u/%02u/%04u\r\n", 
               tChannelInfo.bFWMonth,
               tChannelInfo.bFWDay,
               tChannelInfo.usFWYear);

        ++ulChannel;
      }

      ++ulBoard;
    }

    /* close previously opened driver */
    xDriverClose(hDriver);
  }

  printf(" State = 0x%08X\r\n", (int)lRet);
  printf("----------------------------------------------------\r\n");

  return lRet;
}

/*****************************************************************************/
/*! Function to demonstrate system channel functionality (PacketTransfer)
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t SysdeviceDemo(void)
{
  CIFXHANDLE hDriver = NULL;
  int32_t    lRet    = xDriverOpen(&hDriver);

  printf("\r\n---------- System Device handling demo ----------\r\n");

  if(CIFX_NO_ERROR == lRet)
  {
    /* Driver/Toolkit successfully opened */
    CIFXHANDLE hSys = NULL;
    lRet = xSysdeviceOpen(hDriver, "cifX0", &hSys);

    if(CIFX_NO_ERROR != lRet)
    {
      printf("Error opening SystemDevice!\r\n");

    } else
    {
      SYSTEM_CHANNEL_SYSTEM_INFO_BLOCK tSysInfo = {{0}};
      uint32_t ulSendPktCount = 0;
      uint32_t ulRecvPktCount = 0;
      CIFX_PACKET tSendPkt = {{0}};
      CIFX_PACKET tRecvPkt = {{0}};
      
      /* System channel successfully opened, try to read the System Info Block */
      if( CIFX_NO_ERROR != (lRet = xSysdeviceInfo(hSys, CIFX_INFO_CMD_SYSTEM_INFO_BLOCK, sizeof(tSysInfo), &tSysInfo)))
      {
        printf("Error querying system information block\r\n");
      } else
      {
        printf("System Channel Info Block:\r\n");
        printf("DPM Size         : %u\r\n", (int)tSysInfo.ulDpmTotalSize);
        printf("Device Number    : %u\r\n", (int)tSysInfo.ulDeviceNumber);
        printf("Serial Number    : %u\r\n", (int)tSysInfo.ulSerialNumber);
        printf("Manufacturer     : %u\r\n", (int)tSysInfo.usManufacturer);
        printf("Production Date  : %u\r\n", (int)tSysInfo.usProductionDate);
        printf("Device Class     : %u\r\n", (int)tSysInfo.usDeviceClass);
        printf("HW Revision      : %u\r\n", (int)tSysInfo.bHwRevision);
        printf("HW Compatibility : %u\r\n", (int)tSysInfo.bHwCompatibility);
      }

      /* Do a simple Packet exchange via system channel */
      xSysdeviceGetMBXState(hSys, &ulRecvPktCount, &ulSendPktCount);
      printf("System Mailbox State: MaxSend = %u, Pending Receive = %u\r\n",
             (int)ulSendPktCount, (int)ulRecvPktCount);

      if(CIFX_NO_ERROR != (lRet = xSysdevicePutPacket(hSys, &tSendPkt, 10)))
      {
        printf("Error sending packet to device!\r\n");
      } else
      {
        printf("Send Packet:\r\n");
        DumpPacket(&tSendPkt);

        xSysdeviceGetMBXState(hSys, &ulRecvPktCount, &ulSendPktCount);
        printf("System Mailbox State: MaxSend = %u, Pending Receive = %u\r\n",
              (int)ulSendPktCount, (int)ulRecvPktCount);

        if(CIFX_NO_ERROR != (lRet = xSysdeviceGetPacket(hSys, sizeof(tRecvPkt), &tRecvPkt, 20)) )
        {
          printf("Error getting packet from device!\r\n");
        } else
        {
          printf("Received Packet:\r\n");
          DumpPacket(&tRecvPkt);

          xSysdeviceGetMBXState(hSys, &ulRecvPktCount, &ulSendPktCount);
          printf("System Mailbox State: MaxSend = %u, Pending Receive = %u\r\n",
               (int)ulSendPktCount, (int)ulRecvPktCount);
        }
      }
      xSysdeviceClose(hSys);
    }

    xDriverClose(hDriver);
  }

  printf(" State = 0x%08X\r\n", (int)lRet);
  printf("----------------------------------------------------\r\n");

  return lRet;
}

/*****************************************************************************/
/*! Function to demonstrate communication channel functionality
*   Packet Transfer and I/O Data exchange
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t ChannelDemo(void)
{
  CIFXHANDLE  hDriver = NULL;
  int32_t     lRet    = xDriverOpen(&hDriver);

  printf("\r\n---------- Communication Channel demo ----------\r\n");

  if(CIFX_NO_ERROR == lRet)
  {
    /* Driver/Toolkit successfully opened */
    CIFXHANDLE hChannel = NULL;
    lRet = xChannelOpen(NULL, "cifX0", 0, &hChannel);

    if(CIFX_NO_ERROR != lRet)
    {
      printf("Error opening Channel!");

    } else
    {
      CHANNEL_INFORMATION tChannelInfo = {{0}};
      uint32_t ulCycle = 0;
      uint8_t abSendData[32] = {0};
      uint8_t abRecvData[32] = {0};
      CIFX_PACKET tSendPkt = {{0}};
      CIFX_PACKET tRecvPkt = {{0}};
      
      /* Channel successfully opened, so query basic information */
      if( CIFX_NO_ERROR != (lRet = xChannelInfo(hChannel, sizeof(CHANNEL_INFORMATION), &tChannelInfo)))
      {
        printf("Error querying system information block\r\n");
      } else
      {
        printf("Communication Channel Info:\r\n");
        printf("Device Number    : %u\r\n", (int)tChannelInfo.ulDeviceNumber);
        printf("Serial Number    : %u\r\n", (int)tChannelInfo.ulSerialNumber);
        printf("Firmware         : %s\r\n", tChannelInfo.abFWName);
        printf("FW Version       : %u.%u.%u build %u\r\n", 
                tChannelInfo.usFWMajor,
                tChannelInfo.usFWMinor,
                tChannelInfo.usFWRevision,
                tChannelInfo.usFWBuild);
        printf("FW Date          : %02u/%02u/%04u\r\n", 
                tChannelInfo.bFWMonth,
                tChannelInfo.bFWDay,
                tChannelInfo.usFWYear);

        printf("Mailbox Size     : %u\r\n", (int)tChannelInfo.ulMailboxSize);
      }

      /* Do a basic Packet Transfer */
      if(CIFX_NO_ERROR != (lRet = xChannelPutPacket(hChannel, &tSendPkt, 10)))
      {
        printf("Error sending packet to device!\r\n");
      } else
      {
        printf("Send Packet:\r\n");
        DumpPacket(&tSendPkt);

        if(CIFX_NO_ERROR != (lRet = xChannelGetPacket(hChannel, sizeof(tRecvPkt), &tRecvPkt, 20)) )
        {
          printf("Error getting packet from device!\r\n");
        } else
        {
          printf("Received Packet:\r\n");
          DumpPacket(&tRecvPkt);
        }
      }

      /* Read and write I/O data (32Bytes). Output data will be incremented each cyle */
      xChannelIOReadSendData(hChannel, 0, 0, sizeof(abRecvData), abRecvData);

      for( ulCycle = 0; ulCycle < 10; ++ulCycle)
      {
        if(CIFX_NO_ERROR != (lRet = xChannelIORead(hChannel, 0, 0, sizeof(abRecvData), abRecvData, 10)))
        {
          printf("Error reading IO Data area!\r\n");
          break;
        } else
        {
          printf("IORead Data:");
          DumpData(abRecvData, sizeof(abRecvData));

          if(CIFX_NO_ERROR != (lRet = xChannelIOWrite(hChannel, 0, 0, sizeof(abSendData), abSendData, 10)))
          {
            printf("Error writing to IO Data area!\r\n");
            break;
          } else
          {
            printf("IOWrite Data:");
            DumpData(abSendData, sizeof(abSendData));

            memset(abSendData, (int)ulCycle + 1, sizeof(abSendData));
          }
        }
      }

      xChannelClose(hChannel);
    }

    xDriverClose(hDriver);
  }

  printf(" State = 0x%08X\r\n", (int)lRet);
  printf("----------------------------------------------------\r\n");

  return lRet;

}

/*****************************************************************************/
/*! Function to demonstrate control/status block functionality
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t BlockDemo(void)
{
  int32_t lRet = CIFX_NO_ERROR;
  CIFXHANDLE hDevice = NULL;
    
  printf("\n--- Read / Write Block Information ---\r\n");  

  /* Open channel */
  lRet = xChannelOpen(NULL, "cifX0", 0, &hDevice);
  if(lRet != CIFX_NO_ERROR)
  {
    printf("Error opening Channel!\r\n");
  } else
  {
    uint8_t abBuffer[4] = {0};

    /* Read / Write control block */
    printf("Read CONTROL Block \r\n");  
    memset( abBuffer, 0, sizeof(abBuffer));
    lRet = xChannelControlBlock( hDevice, CIFX_CMD_READ_DATA, 0, 4, &abBuffer[0]);

    DumpData(abBuffer, 4);

    printf("Write CONTROL Block \r\n");  
    lRet = xChannelControlBlock( hDevice, CIFX_CMD_WRITE_DATA, 0, 4, &abBuffer[0]);

    printf("Read COMMON Status Block \r\n");  
    memset( abBuffer, 0, sizeof(abBuffer));
    lRet = xChannelCommonStatusBlock( hDevice, CIFX_CMD_READ_DATA, 0, 4, &abBuffer[0]);

    DumpData(abBuffer, 4);

    printf("Write COMMON Status Block \r\n");  
    lRet = xChannelCommonStatusBlock( hDevice, CIFX_CMD_WRITE_DATA, 0, 4, &abBuffer[0]);

    /* this is expected to fail, as this block must not be written by Host */
    if(CIFX_NO_ERROR != lRet)
      printf("Error writing to common status block. lRet = 0x%08x\r\n", (int)lRet);

    printf("Read EXTENDED Status Block \r\n");  
    memset( abBuffer, 0, sizeof(abBuffer));
    lRet = xChannelExtendedStatusBlock( hDevice, CIFX_CMD_READ_DATA, 0, 4, &abBuffer[0]);
    DumpData(abBuffer, 4);
    
    printf("Write EXTENDED Status Block \r\n");  
    lRet = xChannelExtendedStatusBlock( hDevice, CIFX_CMD_WRITE_DATA, 0, 4, &abBuffer[0]);

    /* this is expected to fail, as this block must not be written by Host */
    if(CIFX_NO_ERROR != lRet)
      printf("Error writing to extended status block. lRet = 0x%08x\r\n", (int)lRet);

    xChannelClose(hDevice);
  } 
  

  return lRet;
}

/*****************************************************************************/
/*! Toolkit Example Task
*   \return 0                                                                */
/*****************************************************************************/
int32_t cifXToolkitInit(void)
{
  int32_t lTkRet = CIFX_NO_ERROR;
    
  /* First of all initialize toolkit */
  lTkRet = cifXTKitInit();
    
  if(CIFX_NO_ERROR == lTkRet)
  {
    PDEVICEINSTANCE ptDevInstance = (PDEVICEINSTANCE)OS_Memalloc(sizeof(*ptDevInstance));
    OS_Memset(ptDevInstance, 0, sizeof(*ptDevInstance));

    /* Set trace level of toolkit */
      g_ulTraceLevel = TRACE_LEVEL_ERROR   | 
                       TRACE_LEVEL_WARNING | 
                       TRACE_LEVEL_INFO    | 
                       TRACE_LEVEL_DEBUG;

    /* Insert the basic device information , into the DeviceInstance structure 
       for the toolkit. The DPM address must be zero, as we only transfer address 
       offsets via the SPI interface.
       NOTE: The physical address and irq number are for information use 
             only, so we skip them here. Interrupt is currently not supported
             and ignored, so we dont need to set it */
    ptDevInstance->fPCICard          = 0;          
    ptDevInstance->pvOSDependent     = ptDevInstance;
    ptDevInstance->ulDPMSize         = 0x10000;
    OS_Strncpy(ptDevInstance->szName,
              "cifX0",
              sizeof(ptDevInstance->szName));

    #if (NX50_PARDPM8 == 1) | (NX50_PARDPM16 == 1)
	    ptDevInstance->pbDPM         = (void*)0x00400000;
      ptDevInstance->pfnHwIfRead   = MQX_ParDPM_Read;
      ptDevInstance->pfnHwIfWrite  = MQX_ParDPM_Write;
    #elif (NX50_SERDPM == 1)
      SerialDPM_Init(ptDevInstance);
    #endif

    /* Add the device to the toolkits handled device list */
    lTkRet = cifXTKitAddDevice(ptDevInstance);

    /* Create polling task */
    _task_create(0, CIFXPOLLING_TASK, 500);
    
    /* If it succeeded do device tests */    
    if(CIFX_NO_ERROR != lTkRet)
    {
      /* Uninitialize Toolkit, this will remove all handled boards from the toolkit and 
         deallocate the device instance */
      cifXTKitDeinit();
    }           
  }
  
  return lTkRet;
}
